<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\PostController;
use App\Http\Controllers\AdminController;
use App\Http\Controllers\ContactController;
use App\Http\Middleware\AdminAuth; // <-- important
use App\Models\Post;
// ----------------- Public/User Routes -----------------

// Homepage - list all posts (with search & pagination)
Route::get('/', [PostController::class, 'index'])->name('posts.index');

// About page (static blade)
Route::view('/about', 'pages.about')->name('about');

Route::view('/terms', 'pages.terms')->name('terms');


// Privacy Policy page (AdSense required)
Route::view('/privacy-policy', 'pages.privacy')->name('privacy.policy');


// Contact page
Route::get('/contact', [ContactController::class, 'show'])->name('contact');

// Handle contact form submit
Route::post('/contact', [ContactController::class, 'send'])->name('contact.send');

// Archive main page (e.g. list years / months)
Route::get('/archive', [PostController::class, 'archive'])->name('archive');

// Archive by year
Route::get('/archive/{year}', [PostController::class, 'archiveYear'])
    ->whereNumber('year')
    ->name('archive.year');

// Archive by year + month
Route::get('/archive/{year}/{month}', [PostController::class, 'archiveMonth'])
    ->whereNumber('year')
    ->whereNumber('month')
    ->name('archive.month');

// Category routes (news, story, lifestyle)
Route::get('/category/{category}', [PostController::class, 'category'])
    ->whereIn('category', ['news', 'story', 'lifestyle'])
    ->name('category.show');

// Show single post by slug
Route::get('/posts/{post:slug}', [PostController::class, 'show'])->name('posts.show');

// Users can comment on posts
Route::post('/posts/{post}/comments', [PostController::class, 'storeComment'])->name('posts.comment');

// Users can like posts
Route::post('/posts/{post}/like', [PostController::class, 'like'])->name('posts.like');

// Sitemap route (XML)
Route::get('/sitemap.xml', function () {
    $posts = Post::latest()->get();

    return response()->view('sitemap', compact('posts'))
        ->header('Content-Type', 'text/xml');
});

// ----------------- Admin Routes -----------------

// Admin login page (shows the form)
Route::get('/admin/login', [AdminController::class, 'login'])->name('admin.login');

// Admin login submit (handles login logic)
Route::post('/admin/login', [AdminController::class, 'authenticate'])->name('admin.login.submit');

// Admin dashboard (protected with AdminAuth middleware class)
Route::get('/admin/dashboard', [AdminController::class, 'dashboard'])
    ->name('admin.dashboard')
    ->middleware(AdminAuth::class);

// Admin logout
Route::post('/admin/logout', [AdminController::class, 'logout'])->name('admin.logout');

// Admin-only post management
Route::middleware(AdminAuth::class)->prefix('admin')->group(function () {
    Route::get('/posts', [PostController::class, 'index'])->name('admin.posts.index');
    Route::get('/posts/create', [PostController::class, 'create'])->name('admin.posts.create');
    Route::post('/posts', [PostController::class, 'store'])->name('admin.posts.store');
    Route::get('/posts/{post}/edit', [PostController::class, 'edit'])->name('admin.posts.edit');
    Route::put('/posts/{post}', [PostController::class, 'update'])->name('admin.posts.update');
    Route::delete('/posts/{post}', [PostController::class, 'destroy'])->name('admin.posts.destroy');
});
